/*
 *  THUMFuncts.c
 *  THUMCtl
 *
 *  Created by Rich Inacker on 9/5/06
 *
 *	Note: This file contains proprietary information.
 *
 *  This file is not to be disclosed to third parties without the express written consent of
 *	Practical Solutions, Inc. (Yankton, SD).
 *
 */

#include <math.h>
#include <stdio.h>

#include "THUMCtl.h"
#include "IntReportDescriptors.h"

/* ---------------------------------------------------------------------------------------------- */
/* Sanity Checkers                                                                                */
#define SHT75_MIN_TEMP_C			-40.0	/* Lowest valid temperature for SHT75 */
#define SHT75_MAX_TEMP_C			123.8	/* Highest valid temperature for SHT75 */
#define SHT75_MIN_RH				0		/* Lowest valid humidity for SHT75 */
#define SHT75_MAX_RH				100		/* Highest valid humidity for SHT75 */
#define DS1822_MIN_TEMP_C			-55.0	/* Lowest valid temperature for external DS1822 */
#define DS1822_MAX_TEMP_C			125.0	/* Highest valid temperature for external DS1822 */
/* ---------------------------------------------------------------------------------------------- */
/* Functions                                                                                      */
extern int SampleTHUM(int *returnValue, unsigned char reportSelector, unsigned char unitAddress);
/* ---------------------------------------------------------------------------------------------- */

void printErrorDescriptor (int descriptor) {
	switch (descriptor) {
		case THUM_ERROR_BADTEMPUNIT:
			fprintf(stderr, "\tTHUM_ERROR_BADTEMPUNIT: Bad temperature unit (not F, f, C, or c)\n");
			break;
		case THUM_ERROR_THUMNOTFOUND:
			fprintf(stderr, "\tTHUM_ERROR_THUMNOTFOUND: THUM device not found\n");
			break;
		case THUM_ERROR_READTIMEOUT:
			fprintf(stderr, "\tTHUM_ERROR_READTIMEOUT: Read operation timed out\n");
			break;
		case THUM_ERROR_WRITEFAILED:
			fprintf(stderr, "\tTHUM_ERROR_WRITEFAILED: Write operation to THUM device failed\n");
			break;
		case THUM_ERROR_READFAILED:
			fprintf(stderr, "\tTHUM_ERROR_READFAILED: Read operation from THUM device failed\n");
			break;
		case THUM_ERROR_RESULTOUTOFRANGE:
			fprintf(stderr, "\tTHUM_ERROR_RESULTOUTOFRANGE: Temp or RH was out of range\n");
			break;
		case THUM_ERROR_PARAMOUTOFRANGE:
			fprintf(stderr, "\tTHUM_ERROR_PARAMOUTOFRANGE: Parameter was out of range\n");
			break;
		case THUM_ERROR_CANTACCESS:
			fprintf(stderr, "\tTHUM_ERROR_CANTACCESS: Can't access the THUM device\n");
			break;
		case THUM_ERROR_NOTSUPPORTED:
			fprintf(stderr, "\tTHUM_ERROR_NOTSUPPORTED: Sensor not supported on this THUM\n");
			break;
		case THUM_ERROR_BADSENSOR:
			fprintf(stderr, "\tTHUM_ERROR_BADSENSOR: THUM Sensor bad or not connected\n");
			break;
		return;
	}
}
/* ------------------------------------------------------------------------------------- */

int GetTemp(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	double myTempC = 0.0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_INTERNAL_T, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	if (THUMdata == SHT75_MISSING) {
		return THUM_ERROR_BADSENSOR;
	}
	
	/* dblTempC = (ReadBuffer * 0.01) - 41.1111 */
	myTempC = (((double) THUMdata) * 0.01) - 41.1111;
	if ((myTempC <= SHT75_MAX_TEMP_C) && (myTempC >= SHT75_MIN_TEMP_C)) {
		*returnValue = myTempC;
	} else {
		result = THUM_ERROR_RESULTOUTOFRANGE;
	}

	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetRH(double *returnValue, double tempC, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	double myTempC = 0.0, dblRH = 0.0, bufvalue = 0.0;
	
	if (tempC == MUST_MEASURE) {
		/* No previous temp reading taken, so let's get one */
		result = GetTemp(&myTempC, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
	} else {
		/* Already got a temp measurement, so use it */
		myTempC = tempC;
	}

	result = SampleTHUM(&THUMdata, THUM_READ_INTERNAL_RH, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	if (THUMdata == SHT75_MISSING) {
		return THUM_ERROR_BADSENSOR;
	}

	bufvalue = ((double) THUMdata);
	dblRH = -4.0 + (0.0405 * bufvalue) + (-0.0000028 * (bufvalue * bufvalue));
	dblRH = (myTempC - 25.0) * (0.01 + (0.00008 * bufvalue)) + dblRH;
	
	if ((dblRH <= SHT75_MAX_RH) && (dblRH >= SHT75_MIN_RH)) {
		*returnValue = dblRH;
	} else {
		result = THUM_ERROR_RESULTOUTOFRANGE;
	}
	
	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetDP(double *returnValue, double tempC, double RH, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	double myTempC = 0.0, myRH = 0.0, logEW = 0.0, myDP = 0.0;

	if (tempC == MUST_MEASURE) {
		/* No previous temp reading taken, so let's get one */
		result = GetTemp(&myTempC, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
	} else {
		/* Already got a temp measurement, so use it */
		myTempC = tempC;
	}
	
	if (RH == MUST_MEASURE) {
		/* No previous RH reading taken, so let's get one */
		result = GetRH(&myRH, myTempC, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
	} else {
		/* Already got an RH measurement, so use it */
		myRH = RH;
	}
	
	if (myRH < 0) {
		/* Should never happen, but make sure we don't raise a nasty floating point exception on log10 */
		return THUM_ERROR_PARAMOUTOFRANGE;
	}
	
	logEW = (0.66077 + 7.5 * myTempC / (237.3 + myTempC) + (log10(myRH) - 2.0));
	myDP = ((0.66077 - logEW) * 237.3) / (logEW - 8.16077);
	*returnValue = myDP;
	
	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetExternalTemp(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	double myTempC = 0.0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_SOCKET2_DALLAS, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	
	if ((THUMdata == DS1822_REMOTE_NOTSUPPORTED_1) || (THUMdata == DS1822_MISSING)) {
		/* First attempt returned missing code, but there's a minute chance this could be a valid reading */
		/* so use the check code to confirm... */
		result = SampleTHUM(&THUMdata, THUM_CHECK_SOCKET2_DALLAS, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
		if (THUMdata == DS1822_REMOTE_NOTSUPPORTED_2) {
			/* The DS1822 is not supported on this THUM */
			return THUM_ERROR_NOTSUPPORTED;
		}
		if (THUMdata == DS1822_MISSING) {
			/* The DS1822 is bad or missing */
			return THUM_ERROR_BADSENSOR;
		}
	}

	myTempC = ((double) THUMdata) / 100.0;
	if ((myTempC <= DS1822_MAX_TEMP_C) && (myTempC >= DS1822_MIN_TEMP_C)) {
		*returnValue = myTempC;
	} else {
		result = THUM_ERROR_RESULTOUTOFRANGE;
	}

	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetSecondaryInternalTemp(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	double myTempC = 0.0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_INTERNAL_DALLAS, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	
	if ((THUMdata == DS1822_INTERNAL_NOTSUPPORTED_1) || (THUMdata == DS1822_MISSING)) {
		/* First attempt returned missing code, but there's a minute chance this could be a valid reading */
		/* so use the check code to confirm... */
		result = SampleTHUM(&THUMdata, THUM_CHECK_INTERNAL_DALLAS, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
		if (THUMdata == DS1822_INTERNAL_NOTSUPPORTED_2) {
			/* The DS1822 is not supported on this THUM */
			return THUM_ERROR_NOTSUPPORTED;
		}
		if (THUMdata == DS1822_MISSING) {
			/* The DS1822 is bad or missing */
			return THUM_ERROR_BADSENSOR;
		}
	}

	myTempC = ((double) THUMdata) / 100.0;
	if ((myTempC <= DS1822_MAX_TEMP_C) && (myTempC >= DS1822_MIN_TEMP_C)) {
		*returnValue = myTempC;
	} else {
		result = THUM_ERROR_RESULTOUTOFRANGE;
	}

	return result;
}

/* ------------------------------------------------------------------------------------- */

int GetIR(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_IR_PROX, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	
	if ((THUMdata == IR_PROX_NOTSUPPORTED_1) || (THUMdata == IR_PROX_MISSING)) {
		/* First attempt returned an errir code, but there's a minute chance this could be a valid reading */
		/* so use the check code to confirm... */
		result = SampleTHUM(&THUMdata, THUM_CHECK_IR_PROX, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
		if (THUMdata == IR_PROX_NOTSUPPORTED_2) {
			/* The IR sensor is not supported */
			return THUM_ERROR_NOTSUPPORTED;
		}
		if (THUMdata == IR_PROX_MISSING) {
			/* The IR sensor is bad or not there */
			return THUM_ERROR_BADSENSOR;
		}

	}

	*returnValue = ((double) THUMdata);

	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetUltrasonic(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_US_PROX, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	
	if ((THUMdata == US_PROX_NOTSUPPORTED_1) || (THUMdata == US_PROX_MISSING)) {
		/* First attempt returned missing code, but there's a minute chance this could be a valid reading */
		/* so use the check code to confirm... */
		result = SampleTHUM(&THUMdata, THUM_CHECK_US_PROX, unitAddress);
		if (result != THUM_ERROR_SUCCESS) {
			return result;
		}
		if (THUMdata == US_PROX_NOTSUPPORTED_2) {
			return THUM_ERROR_NOTSUPPORTED;
		}
		if (THUMdata == US_PROX_MISSING) {
			return THUM_ERROR_BADSENSOR;
		}
	}

	*returnValue = ((double) THUMdata);

	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetSwitch(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_SWITCH_1, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	if (THUMdata == SWITCH_1_NOTSUPPORTED) {
		return THUM_ERROR_NOTSUPPORTED;;
	}

	*returnValue = ((double) THUMdata);

	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetSwitch2(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_SWITCH_2, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	if (THUMdata == SWITCH_2_NOTSUPPORTED) {
		return THUM_ERROR_NOTSUPPORTED;;
	}

	*returnValue = ((double) THUMdata);

	return result;
}
/* ------------------------------------------------------------------------------------- */

int GetSwitch3(double *returnValue, unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	
	result = SampleTHUM(&THUMdata, THUM_READ_SWITCH_3, unitAddress);
	if (result != THUM_ERROR_SUCCESS) {
		return result;
	}
	if (THUMdata == SWITCH_3_NOTSUPPORTED) {
		return THUM_ERROR_NOTSUPPORTED;;
	}

	*returnValue = ((double) THUMdata);

	return result;
}
/* ------------------------------------------------------------------------------------- */

int ResetTHUM(unsigned char unitAddress) {
	int result = THUM_ERROR_SUCCESS;
	int THUMdata = 0;
	
	result = SampleTHUM(&THUMdata, THUM_RESET, unitAddress);

	return result;
}
/* ------------------------------------------------------------------------------------- */

