/*
 *  THUMLib.c
 *
 *  Created by Rich Inacker on 9/18/06.
 *
 *	Updated with dealloc fixes.				10/16/06	RI
 *	Converted to 'persistant r' approach.	10/23/06	RI
 *	Updated and modified for OEM.			07/01//07	RI
 *	x86_64 Version							09/24/11	RI
 *
 *  Rebuilt 64bit only, replaced kIOHIDReportTypeFeature with kIOHIDReportTypeOutput
 *  to make code compatible with USB 1&2 devices - M.Wall Nov 2019
 *
 *	Note: This file contains proprietary information.
 *
 *  This file is not to be disclosed to third parties without the express written consent of
 *	Practical Solutions, Inc. (Yankton, SD).
 *
 */

#include "THUMLibPrivate.h"
#include "THUMLib.h"

/* ------------------------------------------------------------------------------------- */
/* Structures                                                                            */
struct reader {
	io_object_t ioObject;
	IOHIDDeviceInterface122 **interface;
	int gotdata;
	unsigned char buffer[8];
	UInt32 address;
	CFRunLoopSourceRef eventSource;
};

/* ------------------------------------------------------------------------------------- */
/* Private Functions                                                                     */
static void ReaderReportCallback(void *target, IOReturn result, void *refcon, void *sender, UInt32 size) {
	struct reader *r = target;
	r->gotdata = 1;
}

/* ------------------------------------------------------------------------------------- */
/* Public Functions																		*/

int GetTHUMDylibVersion (char *dylibVersionString, int stringSize) {
	if (dylibVersionString != NULL) {
		strlcpy(dylibVersionString, THUM_LIB_VERSION, stringSize);
		return THUM_ERROR_SUCCESS;
	}
	return THUM_ERROR_PARAMOUTOFRANGE;
}

int GetTHUMHandle(void *THUMHandle, UInt32 unitAddress, IOHIDCallbackFunction RemovalReportCallback) {
	/* Note: Unit address is ignored in this implementation.	*/
	struct reader *r = THUMHandle;
	if (r != NULL) {
		IOReturn result = kIOReturnSuccess;
		io_iterator_t hidObjectIterator = 0;
		io_object_t hidDevice = IO_OBJECT_NULL;
		SInt32 score = 0;
		IOCFPlugInInterface **plugInInterface = NULL;
		HRESULT plugInResult = S_OK;
		IOHIDCallbackFunction callback = RemovalReportCallback;
		CFMutableDictionaryRef hidMatchDictionary = NULL;
		CFNumberRef cfVendorID, cfProductID, cfVersionNumber;
		UInt32 value = 0;
		
		/* Create dictionary values for the THUM Vendor, Product IDs and unit address (not currently used) */
		value = THUM_VENDORID;
		cfVendorID = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &value);
		value = THUM_PRODID;
		cfProductID = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &value);
		value = unitAddress;
		r->address = unitAddress;
		cfVersionNumber = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &value);

		/* Start with a dictionary that matches all USB HID devices */
		hidMatchDictionary = IOServiceMatching(kIOHIDDeviceKey);
		
		/* Add keys to narrow down the search to only THUMs */
		CFDictionaryAddValue(hidMatchDictionary, CFSTR(kIOHIDVendorIDKey), cfVendorID);
		CFRelease(cfVendorID);
		CFDictionaryAddValue(hidMatchDictionary, CFSTR(kIOHIDProductIDKey), cfProductID);
		CFRelease(cfProductID);
		/* Note: Future ability to target a specific THUM by unitAddress/VersionNumber */
		/* CFDictionaryAddValue(hidMatchDictionary, CFSTR(kIOHIDVersionNumberKey), cfVersionNumber); */
		CFRelease(cfVersionNumber);

		/* Ask for a interator that will find all THUMs, based on our matching dictionary */
		result = IOServiceGetMatchingServices(kIOMasterPortDefault, hidMatchDictionary, &hidObjectIterator);

		if ((result != kIOReturnSuccess) || (hidObjectIterator == 0)) {
			return THUM_ERROR_THUMNOTFOUND;
		}
		
		/* Get the first THUM in the list */
		hidDevice = IOIteratorNext(hidObjectIterator);
		
		/* Jettison the iterator */
		IOObjectRelease(hidObjectIterator);

		/* Now set up an intermediate "plug-in" interface to the THUM */
		r->ioObject = hidDevice;

		result = IOCreatePlugInInterfaceForService(hidDevice, kIOHIDDeviceUserClientTypeID, kIOCFPlugInInterfaceID, &plugInInterface, &score);
		
		/* Jettison the hidObject, since we have an intermediate interface to it now */
		IOObjectRelease(hidDevice);
		
		if (result != kIOReturnSuccess) {
			IOObjectRelease(r->ioObject);
			return THUM_ERROR_CANTACCESS;
		}
		
		/* Get a direct HID class device interface  */
		plugInResult = (*plugInInterface)->QueryInterface(plugInInterface, CFUUIDGetUUIDBytes(kIOHIDDeviceInterfaceID), (LPVOID) &(r->interface));
		
		/* Jettison the intermediate interface */
		result = IODestroyPlugInInterface(plugInInterface);
		
		if (plugInResult != S_OK) {
			if (r->interface != NULL) {
				(*(r->interface))->Release(r->interface);
			}
			IOObjectRelease(r->ioObject);
			return THUM_ERROR_CANTACCESS;
		}

		/* Now we can open the HID interface to THUM */
		result = (*(r->interface))->open(r->interface, 0);
		if (result != kIOReturnSuccess) {
			(*(r->interface))->Release(r->interface);
			IOObjectRelease(r->ioObject);
			return THUM_ERROR_CANTACCESS;
		}		
			
		/* Set up all the interface housekeeping */
		(*(r->interface))->createAsyncEventSource(r->interface, &r->eventSource);
		(*(r->interface))->setInterruptReportHandlerCallback(r->interface, r->buffer, 8, ReaderReportCallback, r, NULL);
		
		if (callback != NULL) {
			/*This will generate a compiler warning, but OK since we're using refcon to provide unitAddress, an integer, not a pointer to a struct */
			(*(r->interface))->setRemovalCallback(r->interface, RemovalReportCallback, NULL, unitAddress);
		}
				
		(*(r->interface))->startAllQueues(r->interface);
		
		/* Wire the interrupt data read routine into the current program's run loop */
		CFRunLoopAddSource(CFRunLoopGetCurrent(), r->eventSource, kCFRunLoopDefaultMode);
		
		return THUM_ERROR_SUCCESS;
	}
	return THUM_ERROR_PARAMOUTOFRANGE;
}

int ReleaseTHUMHandle(void *THUMHandle) {
	struct reader *r = THUMHandle;

	if (r != NULL) {
		/* Remove the interrupt data read routine from this program's run loop */
		CFRunLoopRemoveSource(CFRunLoopGetCurrent(), r->eventSource, kCFRunLoopDefaultMode);
		CFRelease(r->eventSource);

		/* Tear down the interface */
		(*(r->interface))->stopAllQueues(r->interface);
		(*(r->interface))->close(r->interface);
		(*(r->interface))->Release(r->interface);
		
		IOObjectRelease(r->ioObject);
		
		return THUM_ERROR_SUCCESS;
	}
	return THUM_ERROR_PARAMOUTOFRANGE;
}

int ReadTHUM(void *THUMHandle, double *tempC, double *rh) {
	struct reader *r = THUMHandle;
	if (r != NULL) {
		IOReturn result = kIOReturnSuccess;
		SInt32 reason = 0;
		int loopCount = 0, THUMdata = 0;
		static unsigned char getTemp[2] = {0x00, 0x00};
		static unsigned char getRH[2] = {0x01, 0x00};
		double myTempC = 0.0, dblRH = 0.0, bufvalue = 0.0;
		
		/* Request Current Temperature from THUM and read it back */
		r->gotdata = 0;
		result = (*(r->interface))->setReport(r->interface, kIOHIDReportTypeOutput, 1, getTemp, sizeof(getTemp), 100, 0, 0, 0);
		if (result != kIOReturnSuccess) {
			return THUM_ERROR_WRITEFAILED;
		}
		while (loopCount < READTRIES) {
			reason = CFRunLoopRunInMode(kCFRunLoopDefaultMode, 0.1, false);
			if (r->gotdata) {
				THUMdata = (r->buffer[0] * 256) + r->buffer[1];
				break;
			}
			++loopCount;
		}
		if (loopCount == READTRIES) {
			return THUM_ERROR_READTIMEOUT;
		}
		if (THUMdata == SHT75_MISSING) {
			return THUM_ERROR_BADSENSOR;
		}
		myTempC = (((double) THUMdata) * 0.01) - 41.1111;
		if ((myTempC <= SHT75_MAX_TEMP_C) && (myTempC >= SHT75_MIN_TEMP_C)) {
			*tempC = myTempC;
		} else {
			return THUM_ERROR_RESULTOUTOFRANGE;
		}

		loopCount = 0;
		THUMdata = 0;
		/* Request Current Humidity from THUM and read it back */
		r->gotdata = 0;
		result = (*(r->interface))->setReport(r->interface, kIOHIDReportTypeOutput, 1, getRH, sizeof(getRH), 100, 0, 0, 0);
		if (result != kIOReturnSuccess) {
			return THUM_ERROR_WRITEFAILED;
		}
		while (loopCount < READTRIES) {
			reason = CFRunLoopRunInMode(kCFRunLoopDefaultMode, 0.1, false);
			if (r->gotdata) {
				THUMdata = (r->buffer[0] * 256) + r->buffer[1];
				break;
			}
			++loopCount;
		}
		if (loopCount == READTRIES) {
			return THUM_ERROR_READTIMEOUT;
		}
		if (THUMdata == SHT75_MISSING) {
			return THUM_ERROR_BADSENSOR;
		}
		bufvalue = ((double) THUMdata);
		dblRH = -4.0 + (0.0405 * bufvalue) + (-0.0000028 * (bufvalue * bufvalue));
		dblRH = (myTempC - 25.0) * (0.01 + (0.00008 * bufvalue)) + dblRH;
		if ((dblRH <= SHT75_MAX_RH) && (dblRH >= SHT75_MIN_RH)) {
			*rh = dblRH;
		} else {
			result = THUM_ERROR_RESULTOUTOFRANGE;
		}
		return THUM_ERROR_SUCCESS;
	}
	return THUM_ERROR_PARAMOUTOFRANGE;
}

int ResetTHUM(void *THUMHandle) {
	struct reader *r = THUMHandle;
    if (r != NULL) {
        IOReturn result = kIOReturnSuccess;
        SInt32 reason = 0;
        int loopCount = 0, THUMdata = 0;
        static unsigned char sendReset[2] = {0x03, 0x00};
        
        /* Issue reset instruction to THUM */
        r->gotdata = 0;
        result = (*(r->interface))->setReport(r->interface,            // IOHIDDeviceInterface122
                                              kIOHIDReportTypeOutput,  // reportType
                                              1,                       // reportID
                                              sendReset,               // reportBuffer
                                              sizeof(sendReset),       // reportBufferSize
                                              100,                     // timeoutMS
                                              0,                       // callback
                                              0,                       // callbackTarget
                                              0);                      // callbackRefcon
        if (result != kIOReturnSuccess) {
            return THUM_ERROR_WRITEFAILED;
        }
        while (loopCount < READTRIES) {
            reason = CFRunLoopRunInMode(kCFRunLoopDefaultMode, 0.1, false);
            if (r->gotdata) {
                THUMdata = (r->buffer[0] * 256) + r->buffer[1];
                break;
            }
            ++loopCount;
        }
        return THUM_ERROR_SUCCESS;
    }
	return THUM_ERROR_PARAMOUTOFRANGE;
}

