/*
 *  THUMComm.c -- Handle Mac OS X Device-level USB comms with THUM
 *
 *  Created by Rich Inacker on 9/4/06.
 *
 *	Updated with all dealloc fixes on 10/16/06	RI
 *
 *  Rebuilt 64bit only, replaced kIOHIDReportTypeFeature with kIOHIDReportTypeOutput
 *  to make code compatible with USB 1&2 devices - M.Wall Nov 2019
 */

#include <IOKit/IOCFPlugIn.h>
#include <IOKit/hid/IOHIDKeys.h>
#include <IOKit/hid/IOHIDLib.h>
#include <CoreFoundation/CoreFoundation.h>

#include "THUMCtl.h"

/* ------------------------------------------------------------------------------------- */
/* Various constants and info                                                            */
#define NODATA						0xFFFF
#define THUM_VENDORID				0x0C70
#define THUM_PRODID					0x0750
#define READTRIES					9		/* Number of attempts to read before timeout */

/* ------------------------------------------------------------------------------------- */
/* Structures                                                                            */
struct reader {
	io_object_t ioObject;
	IOHIDDeviceInterface122 **interface;
	int gotdata;
	unsigned char buffer[8];
};

/* ------------------------------------------------------------------------------------- */
/* Private Functions                                                                     */
static void ReaderReportCallback(void *target, IOReturn result, void *refcon, void *sender, UInt32 size) {
	struct reader *r = target;
	r->gotdata = 1;
}

/* ------------------------------------------------------------------------------------- */

int SampleTHUM(int *returnValue, unsigned char reportSelector, unsigned char unitAddress) {

	/* Note: Unit address is ignored in this implementation.	*/
	int THUMdata = NODATA;
	struct reader *r;

	IOReturn result = kIOReturnSuccess;
	HRESULT plugInResult = S_OK;
	SInt32 reason = 0;
	
	CFMutableDictionaryRef hidMatchDictionary = NULL;
	CFNumberRef cfVendorID, cfProductID;
	/* CFNumberRef cfVersionNumber; */
	UInt32 value = 0;

	io_iterator_t hidObjectIterator = 0;
	io_object_t hidDevice = IO_OBJECT_NULL;
	SInt32 score = 0;
	IOCFPlugInInterface **plugInInterface = NULL;
	CFRunLoopRef runLoop = NULL;
	CFRunLoopSourceRef eventSource = NULL;

	int loopCount = 0;
		
	unsigned char intRequestData[2]	= {0x00, 0x00};
	
	/* Create dictionary values for the THUM Vendor and Product IDs */
	value = THUM_VENDORID;
	cfVendorID = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &value);
	value = THUM_PRODID;
	cfProductID = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &value);
    
    /*
	Note: Future ability to targe a specific THUM by unitAddress/VersionNumber
	value = unitAddress;
	cfVersionNumber = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &value);
	*/
    
	/* Start with a dictionary that matches all USB HID devices */
	hidMatchDictionary = IOServiceMatching(kIOHIDDeviceKey);
	
	/* Add keys to narrow down the search to only THUMs */
	CFDictionaryAddValue(hidMatchDictionary, CFSTR(kIOHIDVendorIDKey), cfVendorID);
	CFRelease(cfVendorID);
	CFDictionaryAddValue(hidMatchDictionary, CFSTR(kIOHIDProductIDKey), cfProductID);
	CFRelease(cfProductID);
	/* Note: Future ability to targe a specific THUM by unitAddress/VersionNumber */
	/* CFDictionaryAddValue(hidMatchDictionary, CFSTR(kIOHIDVersionNumberKey), cfVersionNumber); */
	/* CFRelease(cfVersionNumber); */

	/* Ask for a interator that will find all THUMs, based on our matching dictionary */
	result = IOServiceGetMatchingServices(kIOMasterPortDefault, hidMatchDictionary, &hidObjectIterator);
	hidMatchDictionary = NULL;
	
	if ((result != kIOReturnSuccess) || (hidObjectIterator == 0)) {
		return THUM_ERROR_THUMNOTFOUND;
	}
	
	/* Get the first THUM in the list */
	hidDevice = IOIteratorNext(hidObjectIterator);

	/* Jettison the iterator */
	IOObjectRelease(hidObjectIterator);

	/* Now set up an intermediate "plug-in" interface to the THUM */
	r = malloc(sizeof(*r));
	r->ioObject = hidDevice;
	
	result = IOCreatePlugInInterfaceForService(hidDevice, kIOHIDDeviceUserClientTypeID, kIOCFPlugInInterfaceID, &plugInInterface, &score);

	/* Jettison the hidDevice, since we have an intermediate interface to it now */
	IOObjectRelease(hidDevice);

	if (result != kIOReturnSuccess) {
		IOObjectRelease(r->ioObject);
		free(r);
		return THUM_ERROR_CANTACCESS;
	}
		
	/* Get a direct HID class device interface  */
	plugInResult = (*plugInInterface)->QueryInterface(plugInInterface, CFUUIDGetUUIDBytes(kIOHIDDeviceInterfaceID), (LPVOID) &(r->interface));
	
	/* Jettison the intermediate interface */
	result = IODestroyPlugInInterface(plugInInterface);

	if (plugInResult != S_OK) {
		if (r->interface != NULL) {
			(*(r->interface))->Release(r->interface);
		}
		IOObjectRelease(r->ioObject);
		free(r);
		return THUM_ERROR_CANTACCESS;
	}

	/* Now we can open the HID interface to THUM */
	result = (*(r->interface))->open(r->interface, 0);

	if (result != kIOReturnSuccess) {
		(*(r->interface))->Release(r->interface);
		IOObjectRelease(r->ioObject);
		free(r);
		return THUM_ERROR_CANTACCESS;
	}		
		
	/* Set up all the interface housekeeping */
	(*(r->interface))->createAsyncEventSource(r->interface, &eventSource);
	(*(r->interface))->setInterruptReportHandlerCallback(r->interface, r->buffer, 8, ReaderReportCallback, r, 0);
	(*(r->interface))->startAllQueues(r->interface);
	
	
	/* Wire the interrupt data read routine into this program's run loop */
	runLoop = CFRunLoopGetCurrent();
	CFRunLoopAddSource(runLoop, eventSource, kCFRunLoopDefaultMode);
		
	if (reportSelector > 0xEF) {
		/* This is an error check selector, so handle accordingly */
		intRequestData[0] = (reportSelector - 0xF0);
		intRequestData[1] = 0x01;
	} else {
		intRequestData[0] = reportSelector;
	}
	
	/* Debug 
	printf ("\tSending data: %02x %02x\n", intRequestData[0], intRequestData[1]); */
	
	/* Request the value from THUM and read it back */
	r->gotdata = 0;
	result = (*(r->interface))->setReport(r->interface, kIOHIDReportTypeOutput, 1, intRequestData, sizeof(intRequestData), 100, 0, 0, 0);
	if (result != kIOReturnSuccess) {
		(*(r->interface))->stopAllQueues(r->interface);
		(*(r->interface))->close(r->interface);
		(*(r->interface))->Release(r->interface);
		IOObjectRelease(r->ioObject);
		free(r);
		return THUM_ERROR_WRITEFAILED;
	}
	
	while (loopCount < READTRIES) {
		reason = CFRunLoopRunInMode(kCFRunLoopDefaultMode, 0.1, false);
		if (r->gotdata) {
			THUMdata = (r->buffer[0] * 256) + r->buffer[1];
			break;
		}
		++loopCount;
	}
	
	/* Debug
	printf ("\tReceived data: %02x %02x\n", r->buffer[0], r->buffer[1]);  */
	
	if (loopCount == READTRIES) {
		(*(r->interface))->stopAllQueues(r->interface);
		(*(r->interface))->close(r->interface);
		(*(r->interface))->Release(r->interface);
		IOObjectRelease(r->ioObject);
		free(r);
		return THUM_ERROR_READTIMEOUT;
	}

	CFRunLoopRemoveSource(runLoop, eventSource, kCFRunLoopDefaultMode);
	CFRelease(eventSource);
	
	/* Tear down the interface */
	(*(r->interface))->stopAllQueues(r->interface);
	(*(r->interface))->close(r->interface);
	(*(r->interface))->Release(r->interface);
	
	IOObjectRelease(r->ioObject);
	free(r);
	
	*returnValue = THUMdata;

	return THUM_ERROR_SUCCESS;
	
}
