#include <stdlib.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>

#include "THUMCtl.h"

int main (int argc, char **argv) {
/* parse args */
	opterr = 0;
	
	int opt;
	int result = THUM_ERROR_SUCCESS;
	int verbose = THUM_FALSE;
	static int unitAddress = 0; /* Not currently used (always 0) but here for future expansion */

	int H = 0, V = 0, v = 0, f = 0, c = 0, t = 0, h = 0, d = 0, s = 0, s2 = 0, s3 = 0, x = 0, i = 0,
		u = 0, R = 0, T = 0, e = 0;

	double tValue = 0.0, hValue = 0.0, dValue = 0.0, sValue = 0.0, s2Value = 0.0, s3Value = 0.0,
		   xValue = 0.0, iValue = 0.0, uValue = 0.0, TValue = 0.0;

	char tempUnit = 'C';
	char *distanceUnit;
	
	while ((opt = getopt(argc, argv, "HVRvfctThds123xiue")) != -1) {
		switch (opt) {
			case 'H':
				/* print usage */
				H = 1;
				break;
			case 'V':
				/* print version */
				V = 1;
				break;
			case 'R':
				/* Reset THUM */
				R = 1;
				break;
			case 'v':
				/* toggle verbosity on */
				v = 1;
				break;
			case 'f':
				/* set degrees f */
				f = 1;
				break;
			case 'c':
				/* set degrees c */
				c = 1;
				break;
			case 'e':
				/* enumerate switch values as 'open' or 'closed' */
				e = 1;
				break;
			case 't':
				/* display temp from internal temp/humidity sensor */
				t = 1;
				break;
			case 'T':
				/* display temp from internal temp only sensor */
				T = 1;
				break;
			case 'h':
				/* display humidity from internal temp/humidity sensor */
				h = 1;
				break;
			case 'd':
				/* display dew point */
				d = 1;
				break;
			case 's':
				/* display switch value on socket 1 */
				s = 1;
				break;
			case '1':
				/* display switch value on socket 1 */
				s = 1;
				break;
			case '2':
				/* display switch2 value */
				s2 = 1;
				break;
			case '3':
				/* display switch3 value */
				s3 = 1;
				break;
			case 'x':
				/* display external temp */
				x = 1;
				break;
			case 'i':
				/* display IR reading */
				i = 1;
				break;
			case 'u':
				/* display ultrasonic reading */
				u = 1;
				break;
			default:
				fprintf(stderr, "Unknown option specified\n");
				return 1;
		}
	}	

	/* If -H or -V is supplied, display requested information and exit */
	if(H == 1){
		printf("Usage:\n");
		printf("thumctl <options> where <options> is one or more of the following:\n");
		printf("\t-H  Print this message.\n");
		printf("\t-V  Print THUM library version.\n");
		printf("\t-R  Reset THUM hardware.\n");
		printf("\t-v  Use verbose output.\n");
		printf("\t-f  Display temperatures in degrees F and/or ultrasonic proximities in inches.\n");
		printf("\t-c  Display temperatures in degrees C and/or ultrasonic proximities in centimeters.\n");
		printf("\t-t  Take a temperature reading from internal temp/humidity sensor.\n");
		printf("\t-T  Take a temperature reading from secondary internal temperature sensor (THUM OEM Only).\n");
		printf("\t-h  Take a relative humidity reading.\n");
		printf("\t-d  Take a dew point reading.\n");
		printf("\t-s  Take a switch reading from switch on socket 1 (THUM OEM Only).\n");
		printf("\t-1  Take a switch reading from switch on socket 1...synonym for -s (THUM OEM Only).\n");
		printf("\t-2  Take a switch reading from switch on socket 2 (THUM OEM Only).\n");
		printf("\t-3  Take a switch reading from switch on socket 3 (THUM OEM Only).\n");
		printf("\t-e  Report/enumerate switch readings as 'open' or 'closed' (THUM OEM Only).\n");
		printf("\t-x  Take an external temperature reading. (THUM OEM Only).\n");
		printf("\t-i  Take an IR proximity reading Displays raw reading (THUM OEM Only).\n");
		printf("\t-u  Take an ultrasonic reading. Displays raw reading unless -f or -c specified (THUM OEM Only).\n");
		printf("\n\tSpecifying no options will result in all sensors being sampled and results reported in degrees C/centimeters.\n");
		printf("\tNote: This may fail if sensors are not present or not supported on the connected THUM device.");
		printf("\n\n");
		return 0;
	}

	if(V == 1) {
		printf("thumctl/mac (Xcode 3.2.6 Intel 64 bit) version %s\n", THUMCTL_MAC_VERSION);
		return 0;
	}
    
	/* Otherwise perform requested tests */
	
	if(v == 1){
		verbose = THUM_TRUE;
		printf("Enabling verbose output\n");
	}

	/* if no args specified, read all internal sensors, using degrees C */
	if(argc == 1){
		c = t = T = h = d = s = x = i = u = 1;
	}
	
	if (f == 1) {
		if(verbose) {
			printf("Setting temperature unit to F\n");
		}
		tempUnit = 'F';
	}
	
	if(c == 1){
		if(verbose){
			printf("Setting temperature unit to C\n");
		}
		tempUnit = 'C';
	}

	if(e == 1){
		if(verbose){
			printf("Enumerating switch values as open/closed\n");
		}
	}

	if(t == 1){
		if(verbose){
			printf("Taking internal temperature reading\n");
		}
		/* Just take everything in degC at this point, convert later if necessary */
		result = GetTemp(&tValue, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Internal Temperature Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}

	if(h == 1){
		if(verbose){
			printf("Taking relative humidity reading\n");
		}
		if (t==1) {
			/* We already have the temp reading, so pass it along */
			result = GetRH(&hValue, tValue, unitAddress);
		} else {
			/* No temp reading taken, so GetRH will have to take one */
			result = GetRH(&hValue, MUST_MEASURE, unitAddress);
		}
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr,"Internal relative humidity reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}

	if(d == 1){
		if(verbose){
			printf("Taking dew point reading\n");
		}
		if ((t==1) && (h==1)) {
			/* Already have temp and RH data, so pass it along */
			result = GetDP(&dValue, tValue, hValue, unitAddress);
		} else {
			if (t==1) {
				/* Missing an RH reading, but have temp, so tell GetDP to take RH */
				result = GetDP(&dValue, tValue, MUST_MEASURE, unitAddress);
			} else if (h==1) {
				/* Missing a temp reading, but have RH so tell GetDP to take temp */
				result = GetDP(&dValue, MUST_MEASURE, hValue, unitAddress);
			} else {
				/* We've got nada, so GetDP has to do all the work itself */
				result = GetDP(&dValue, MUST_MEASURE, MUST_MEASURE, unitAddress);
			}
		}
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr,"Internal dew point reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}
	
	if(s == 1){
		if(verbose){
			printf("Taking switch1 reading\n");
		}
		result = GetSwitch(&sValue, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Switch Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}
		
	if(s2 == 1){
		if(verbose){
			printf("Taking switch2 reading\n");
		}
		result = GetSwitch2(&s2Value, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Switch 2 Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}

	if(s3 == 1){
		if(verbose){
			printf("Taking switch3 reading\n");
		}
		result = GetSwitch3(&s3Value, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Switch 3 Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}

	if(x == 1){
		if(verbose){
			printf("Taking external temperature reading\n");
		}
		/* Just take everything in degC at this point, convert later if necessary */
		result = GetExternalTemp(&xValue, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "External Temperature Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}
		
	if(T == 1){
		if(verbose){
			printf("Taking secondary internal temperature reading\n");
		}
		/* Just take everything in degC at this point, convert later if necessary */
		result = GetSecondaryInternalTemp(&TValue, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Secondary Internal Temperature Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}

	if(i == 1){
		if(verbose){
			printf("Taking infrared proximity reading\n");
		}
		result = GetIR(&iValue, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Infrared Proximity Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}
	
	if(u == 1){
		if(verbose){
			printf("Taking ultrasonic proximity reading\n");
		}
		result = GetUltrasonic(&uValue, unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "Ultrasonic Proximity Reading FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
		if (f == 1) {
			/* express result in inches */
			distanceUnit = "in";
			uValue = -0.1730805 + (0.011227463 * uValue);
		} else if (c == 1) {
			/* express result in centimeters */
			distanceUnit = "cm";
			uValue = 2.54 * (-0.1730805 + (0.011227463 * uValue));
		}
	}

	if(R == 1){
		if(verbose){
			printf("Resetting THUM hardware\n");
		}
		result = ResetTHUM(unitAddress);
		if(result != THUM_ERROR_SUCCESS) {
			fprintf(stderr, "THUM Reset FAILED!\n");
			if(verbose) {
				printErrorDescriptor(result);
			}
			return 1;
		}
	}

	/* If temp readings were requested in degF, convert everything before printing it out */
	/* We could define a function for CtoF, but probably quicker to just do it inline */
	/* and avoid the function call overhead */
	if (tempUnit == 'F') {
		tValue = (tValue * 1.8) + 32.0;
		TValue = (TValue * 1.8) + 32.0;
		dValue = (dValue * 1.8) + 32.0;
		xValue = (xValue * 1.8) + 32.0;
	}

	/* Print a header */
	time_t tt;
    struct tm *tod;
    time(&tt);
    tod = localtime(&tt);
	
	printf("%d/%d/%d ", tod->tm_mon+1, tod->tm_mday, tod->tm_year+1900);
    if(tod->tm_hour < 10){
		printf("0");
    }
    printf("%d:", tod->tm_hour);
    if(tod->tm_min < 10){
        printf("0");
    }
    printf("%d:", tod->tm_min);
    if(tod->tm_sec < 10){
        printf("0");
    }
    printf("%d ", tod->tm_sec);

	if(t == 1){
		printf("%f %c ", tValue, tempUnit);
	}
	if(T == 1){
		printf("%f %c ", TValue, tempUnit);
	}
	if(h == 1){
		printf("%f ", hValue);
	}
	if(d == 1){
		printf("%f %c ", dValue, tempUnit);
	}
	if(s == 1){
		if (e == 1) {
			if (sValue > 0.9) {
				printf("open ");
			} else {
				printf("closed ");
			}
		} else {
			printf("%f ", sValue);
		}
	}
	if(s2 == 1){
		if (e == 1) {
			if (s2Value > 0.9) {
				printf("open ");
			} else {
				printf("closed ");
			}
		} else {
			printf("%f ", s2Value);
		}
	}
	if(s3 == 1){
		if (e == 1) {
			if (s3Value > 0.9) {
				printf("open ");
			} else {
				printf("closed ");
			}
		} else {
			printf("%f ", s3Value);
		}
	}
	if(x == 1){
		printf("%f %c ", xValue, tempUnit);
	}
	if(i == 1){
		printf("%f ", iValue);
	}
	if(u == 1){
		printf("%f %s", uValue, distanceUnit);
	}
	printf("\n");


	/* done */
	return 0;
}
